// This may look like C code, but it is really -*- C++ -*-
// RTSS --- Railway Total System Simulator
// (c) TAKAGI Ryo (at Kogakuin University)
// RS_Dijkstra.h --- class RS_Dijkstra and related classes, solving the
//  shortest paths problem.
// -----
// ChangeLog:
// 2009. 10. 16
//  The "directed graph" version complete.
// -----

// ----
// Dijkstra's algorithm
// ダイクストラ法
// ----

#ifndef RS_DIJKSTRA_HH
#define RS_DIJKSTRA_HH

#include <map>
#include <iostream>
#include <vector>
#include <deque>
#include "RS_FHeap.hh"

using std :: cerr ;
using std :: endl ;
using std :: vector ;
using std :: deque ;
using std :: map ;
using std :: ostream ;
using std :: string ;

#define RS_DIJKSTRA_PRINT
#define RS_DIJKSTRA_PRINT_FUNC



// ----
// Forward declarations
// ----
template < class KC , class NID >
class RS_DKBranch ;

template < class KC , class NID >
class RS_DKNode ;

template < class KD , class NIX >
std :: ostream & operator<<
( std :: ostream & ostr ,
  RS_DKNode < KD , NIX > const & dkn ) ;

// ----
// Dijkstra's Algorithm: the "node" of the graph
// ダイクストラのアルゴリズム: グラフの「ノード」
// ----
template < class KC , class NID >
class RS_DKNode
{

  template < class KD , class NIX >
  friend std :: ostream & operator<<
  ( std :: ostream & ostr , RS_DKNode < KD , NIX > const & dkn ) ;

private :

  string _name ;
  NID _nid ;
  typename RS_FHeap < KC , RS_DKNode < KC , NID > > :: Entry * _heapent ;
  RS_DKBranch < KC , NID > * _dkb_in ;

  // ----
  // Note: _blist_in contains branches coming into the node. Therefore, they
  // must have this node as "_n_out". Vice versa for _blist_out.
  // 注意: _blist_in はこのノードに「入ってくる」ブランチを保有する。従って，
  // このノードはそのブランチの _n_out となる。_blist_out はその逆。
  // ----
  vector < RS_DKBranch < KC , NID > * > _blist_in ;
  vector < RS_DKBranch < KC , NID > * > _blist_out ;

  mutable typename vector < RS_DKBranch < KC , NID > * > :: size_type
  _iter ;
  mutable bool _iter_to_start ;

  bool _marked ;
  bool _initial ;

  explicit RS_DKNode ( RS_DKNode < KC , NID > const & ) ;

public :
  explicit RS_DKNode ( NID const & _nid_in , string const & _n_in )
    : _nid ( _nid_in ) , _name ( _n_in ) , _heapent ( 0 ) , _dkb_in ( 0 ) ,
      _blist_in () , _blist_out () , _iter ( 0 ) , _iter_to_start ( true ) ,
      _marked ( false ) , _initial ( false ) {}
  void setHeapEntry
  ( typename RS_FHeap < KC , RS_DKNode < KC , NID > > :: Entry * x )
  { _heapent = x ; }

  // ----
  // Functions setting branches in the graph initialisation process.
  // グラフの初期化プロセスにおいてブランチをセットする関数.
  // ----
  void setIncomingBranch ( RS_DKBranch < KC , NID > * x )
  { _blist_in . push_back ( x ) ; }
  void setOutgoingBranch ( RS_DKBranch < KC , NID > * x )
  { _blist_out . push_back ( x ) ; }

  // ----
  // Setting current incoming path, used to follow the shortest path tree.
  // 現在の最短経路が「入ってくる」ブランチ。最短経路木探索のため用いられる。
  // ----
  void setIncomingPath ( RS_DKBranch < KC , NID > * x )
  { _dkb_in = x ; }

  void setAsMarked () { _marked = true ; }

  // ----
  // Set as initial node
  // 始点ノードとセット
  // ----
  void setAsInitial () { _initial = true ; }

  typename RS_FHeap < KC , RS_DKNode < KC , NID > > :: Entry *
  getHeapEntry () const { return _heapent ; }
  RS_DKBranch < KC , NID > * getIncomingPath () const
  { return _dkb_in ; }

  NID const & getNodeID () const { return _nid ; }
  string const & getName () const { return _name ; }
  bool isMarked () const { return _marked ; }
  bool isUnreachable () const { return ! ( _initial || _dkb_in ) ; }

  RS_DKBranch < KC , NID > * getNextBranch () const
  {
    while ( true )
    {
      if ( _iter_to_start )
      {
        _iter_to_start = false ;
      }
      else
      {
        ++ _iter ;
      }
      if ( _iter >= _blist_out . size () )
        return 0 ;
      if ( _blist_out [ _iter ] == _dkb_in )
      {
        continue ;
      }
      return _blist_out [ _iter ] ;
    }
  }
} ;



template < class KD , class NIX >
ostream &
operator<<
( ostream & ostr ,
  RS_DKNode < KD , NIX > const & dkn )
{
  ostr << dkn . getName () ;
  return ostr ;
}



// ----
// Declaration of template class RS_DKBranch
// RS_DKBranch テンプレートクラスの宣言
// ----
template < class KC , class NID >
class RS_DKBranch
{

private :
  KC _weight ;

  // ----
  // Direction: in -> out. Node _n_in must have this branch in its
  // _blist_out. Node _n_out must have this branch in its _blist_in.
  // 方向: in -> out. このブランチは _n_in ノードの _blist_out に，そして
  // _n_out ノードの _blist_in に，それぞれ含まれなければならない。
  // ----
  RS_DKNode < KC , NID > * _n_in ;
  RS_DKNode < KC , NID > * _n_out ;

public :

  explicit RS_DKBranch
  ( RS_DKNode < KC , NID > * _in , RS_DKNode < KC , NID > * _out , KC x )
    : _weight ( x ) , _n_in ( _in ) , _n_out ( _out )
  { _in -> setOutgoingBranch ( this ) ; _out -> setIncomingBranch ( this ) ; }

  RS_DKNode < KC , NID > * getOpposite
  ( RS_DKNode < KC , NID > * x )
  { if ( _n_in == x ) return _n_out ; if ( _n_out == x ) return _n_in ;
    cerr << "Error: could not find opposite node" << endl ; exit ( 1 ) ; }

  KC const & getWeight () const { return _weight ; }

} ;



// ----
// Declaration of template class RS_Dijkstra
// RS_Dijkstra テンプレートクラスの宣言
// ----
template < class KC , class NID >
class RS_Dijkstra
{

  // "Large number" for initialising a node.
  KC _large ;

  // Heap
  RS_FHeap < KC , RS_DKNode < KC , NID > > _hp ;

  // Vector container of RS_DKNode object instances
  // *** May have to consider using map container instead of vector ***
  map < NID const , RS_DKNode < KC , NID > * > _nlist ;

  // Vector container of RS_DKBranch object instances
  // *** May have to consider using map container instead of vector ***
  vector < RS_DKBranch < KC , NID > * > _blist ;

  bool _has_not_run ;

public :

  explicit RS_Dijkstra ( KC const & _l_in )
    : _large ( _l_in ) , _hp () , _nlist () , _blist () ,
      _has_not_run ( true )
  {}

  ~RS_Dijkstra ()
  {
    for ( typename vector < RS_DKBranch < KC , NID > * > :: iterator
            ii = _blist . begin () ; ii != _blist . end () ; ++ ii )
    {
      delete ( * ii ) ;
    }
    for ( typename map < NID const , RS_DKNode < KC , NID > * >
            :: iterator ii = _nlist . begin () ;
          ii != _nlist . end () ; ++ ii )
    {
      typename RS_FHeap < KC , RS_DKNode < KC , NID > > :: Entry *
        iix = ii -> second -> getHeapEntry () ;
      delete iix ;
      delete ii -> second ;
    }
  }

  void setNode ( NID const & _n_in , string const & _name_in )
  {
    RS_DKNode < KC , NID > *
      _n = new RS_DKNode < KC , NID > ( _n_in , _name_in ) ;
    _nlist [ _n_in ] = _n ;
    typename RS_FHeap < KC , RS_DKNode < KC , NID > > :: Entry * _e
      = new typename RS_FHeap < KC , RS_DKNode < KC , NID > > :: Entry
      ( _large , _n , & _hp ) ;
    _n -> setHeapEntry ( _e ) ;
    _hp . insert ( _e ) ;
  }

  RS_DKNode < KC , NID > * getNode
  ( NID const & _n_in )
  {
    typename map < NID const , RS_DKNode < KC , NID > * > :: iterator
      _i_x = _nlist . find ( _n_in ) ;
    if ( _i_x == _nlist . end () )
    {
      cerr << "Error: no node with ID \"" << _n_in
           << "\" in class RS_Dijkstra" << endl ;
      exit ( 1 ) ;
    }
    return _i_x -> second ;
  }

  void setBranch
  ( NID const & _nn_f , NID const & _nn_t , KC const & _wt )
  {
    RS_DKNode < KC , NID > * _n_f = getNode ( _nn_f ) ;
    RS_DKNode < KC , NID > * _n_t = getNode ( _nn_t ) ;
    RS_DKBranch < KC , NID > *
      _bx = new RS_DKBranch < KC , NID > ( _n_f , _n_t , _wt ) ;
    _blist . push_back ( _bx ) ;
  }

  void setBidirectionalBranches
  ( NID const & _nn_f , NID const & _nn_t , KC const & _wt )
  {
    setBranch ( _nn_f , _nn_t , _wt ) ;
    setBranch ( _nn_t , _nn_f , _wt ) ;
  }

  void run ( NID const & _start_nid , KC const & _kc_zero )
  {
    RS_DKNode < KC , NID > * _n_st = getNode ( _start_nid ) ;
#ifdef RS_DIJKSTRA_PRINT
    cerr << "Dijkstra's Algorithm: starting from node \""
         << _n_st -> getHeapEntry () -> getEntryData () -> getName ()
         << "\"" << endl ;
#endif
    _n_st -> setAsInitial () ;
    _n_st -> getHeapEntry () -> decreaseKey ( _kc_zero ) ;
    typename RS_FHeap < KC , RS_DKNode < KC , NID > > :: Entry *
      _min = _hp . deleteMinimum () ;
    _min -> getEntryData () -> setAsMarked () ;
#ifdef RS_DIJKSTRA_PRINT
    cerr << "delete min, node = " << _min -> getEntryData () -> getName ()
         << ", key = " << _min -> getKey () << endl ;
#endif

    while ( _hp . size () )
    {
      while ( RS_DKBranch < KC , NID > *
              _bx = _min -> getEntryData () -> getNextBranch () )
      {
        RS_DKNode < KC , NID > *
          _op = _bx -> getOpposite ( _min -> getEntryData () ) ;
        if ( _op -> isMarked () )
        {
          continue ;
        }
        KC _nkey = _min -> getKey () + _bx -> getWeight () ;
#ifdef RS_DIJKSTRA_PRINT
        cerr << "decrease key, node = " << _op -> getName () << ", key "
             << _op -> getHeapEntry () -> getKey () << " -> " << _nkey ;
#endif
        if ( _nkey < _op -> getHeapEntry () -> getKey () )
        {
#ifdef RS_DIJKSTRA_PRINT
          cerr << ", perform" << endl ;
#endif
          _op -> getHeapEntry () -> decreaseKey ( _nkey ) ;
          _op -> setIncomingPath ( _bx ) ;
        }
        else
        {
#ifdef RS_DIJKSTRA_PRINT
          cerr << ", not perform" << endl ;
#endif
        }
      }
      _min = _hp . deleteMinimum () ;
      _min -> getEntryData () -> setAsMarked () ;
#ifdef RS_DIJKSTRA_PRINT
      cerr << "delete min, node = " << _min -> getEntryData () -> getName ()
           << ", key = " << _min -> getKey () << endl ;
#endif
    }
    _has_not_run = false ;
  }

  deque < NID > getRoute ( NID const & _dest_nid )
  {
    deque < NID > ret_val ;
    ret_val . clear () ;
    if ( _has_not_run )
    {
      cerr << "Error: has not run the Dijkstra's algorithm yet" << endl ;
      exit ( 1 ) ;
    }
    typename map < NID const , RS_DKNode < KC , NID > * > :: const_iterator
      ii = _nlist . find ( _dest_nid ) ;
    if ( ii == _nlist . end () )
    {
      cerr << "Error: destination not found" << endl ;
      exit ( 1 ) ;
    }
    RS_DKNode < KC , NID > * _n = ii -> second ;
    if ( _n -> isUnreachable () )
    {
      return ret_val ;
    }
    ret_val . push_front ( _n -> getNodeID () ) ;
    while ( RS_DKBranch < KC , NID > * _bi = _n -> getIncomingPath () )
    {
      _n = _bi -> getOpposite ( _n ) ;
      ret_val . push_front ( _n -> getNodeID () ) ;
    }
    return ret_val ;
  }

#ifdef RS_DIJKSTRA_PRINT_FUNC
  void printRoute ( NID const & _dest_nid )
  {
    if ( _has_not_run )
    {
      cerr << "Error: has not run the Dijkstra's algorithm yet" << endl ;
      exit ( 101 ) ;
    }
    typename map < NID const , RS_DKNode < KC , NID > * > :: const_iterator
      ii = _nlist . find ( _dest_nid ) ;
    if ( ii == _nlist . end () )
    {
      cerr << "Error: destination not found" << endl ;
      exit ( 1 ) ;
    }
    RS_DKNode < KC , NID > * _n = ii -> second ;
    cerr << "Node \"" << _n -> getName () << "\": " ;
    if ( _n -> isUnreachable () )
    {
      cerr << "unreachable" << endl ;
      return ;
    }
    cerr << _n -> getHeapEntry () -> getKey () << ": "
         << _n -> getName () ;
    while ( RS_DKBranch < KC , NID > * _bi = _n -> getIncomingPath () )
    {
      cerr << "(" << _bi -> getWeight () << ")" ;
      _n = _bi -> getOpposite ( _n ) ;
      cerr << _n -> getName () ;
    }
    cerr << endl ;
  }
#endif
} ;



#endif // RS_DIJKSTRA_HH
