// -*- C++ -*-

#include <iostream>
#include <cstdlib>
#include <vector>
#include <string>
#include "reflist2-BookEntry.h"

using std :: ostream ;
using std :: vector ;
using std :: cerr ;
using std :: endl ;
using std :: string ;


// ----
// Constructor of class BookEntry.
// BookEntry クラスのコンストラクタ.
// ----
BookEntry :: BookEntry
( const XMLNodePointer & n_in )
  : authors ()
{
  bool sw_title = false ;
  bool sw_ym = false ;
  bool sw_pb = false ;
  bool sw_pa = false ;
  for ( const TiXmlNode * n_ch = n_in -> FirstChild () ;
        n_ch ; n_ch = n_ch -> NextSibling () )
  {
    if ( n_ch -> Type () != TiXmlNode :: ELEMENT )
    {
      continue ;
    }
    if ( string ( n_ch -> Value () ) == "title" )
    {
      if ( sw_title )
      {
        cerr << "Error: two or more <title> tags found in <book_entry>."
             << endl ;
        exit ( 1 ) ;
      }
      sw_title = true ;
      bool sw_tf = false ;
      for ( const TiXmlAttribute * t_a
              = n_ch -> ToElement () -> FirstAttribute () ;
            t_a ; t_a = t_a -> Next () )
      {
        if ( string ( t_a -> Name () ) == "name" )
        {
          if ( sw_tf )
          {
            cerr << "Error: duplicate \"name\" attribute to tag <title>"
                 << "... internal error?" << endl ;
            exit ( 1 ) ;
          }
          sw_tf = true ;
          title_str = t_a -> ValueStr () ;
        }
        else
        {
          cerr << "Error: attribute \"" << t_a -> Name ()
               << "\" not allowed in tag <title>" << endl ;
          exit ( 1 ) ;
        }
      }
    }
    else if ( string ( n_ch -> Value () ) == "author" )
    {
      Author * a_x = new Author ( n_ch ) ;
      authors . push_back ( a_x ) ;
    }
    else if ( string ( n_ch -> Value () ) == "publisher" )
    {
      if ( sw_pb )
      {
        cerr << "Error: two or more <publisher> tags found in <book_entry>."
             << endl ;
        exit ( 1 ) ;
      }
      sw_pb = true ;
      bool sw_bf = false ;
      for ( const TiXmlAttribute * t_a
              = n_ch -> ToElement () -> FirstAttribute () ;
            t_a ; t_a = t_a -> Next () )
      {
        if ( string ( t_a -> Name () ) == "name" )
        {
          if ( sw_bf )
          {
            cerr << "Error: duplicate \"name\" attribute to tag <publisher>"
                 << "... internal error?" << endl ;
            exit ( 1 ) ;
          }
          sw_bf = true ;
          publisher_str = t_a -> ValueStr () ;
        }
        else
        {
          cerr << "Error: attribute \"" << t_a -> Name ()
               << "\" not allowed in tag <publisher>" << endl ;
          exit ( 1 ) ;
        }
      }
    }
    else if ( string ( n_ch -> Value () ) == "address" )
    {
      if ( sw_pa )
      {
        cerr << "Error: two or more <address> tags found in <book_entry>."
             << endl ;
        exit ( 1 ) ;
      }
      sw_pa = true ;
      bool sw_af = false ;
      for ( const TiXmlAttribute * t_a
              = n_ch -> ToElement () -> FirstAttribute () ;
            t_a ; t_a = t_a -> Next () )
      {
        if ( string ( t_a -> Name () ) == "name" )
        {
          if ( sw_af )
          {
            cerr << "Error: duplicate \"name\" attribute to tag <address>"
                 << "... internal error?" << endl ;
            exit ( 1 ) ;
          }
          sw_af = true ;
          paddr_str = t_a -> ValueStr () ;
        }
        else
        {
          cerr << "Error: attribute \"" << t_a -> Name ()
               << "\" not allowed in tag <address>" << endl ;
          exit ( 1 ) ;
        }
      }
    }
    else if ( string ( n_ch -> Value () ) == "year_month" )
    {
      if ( sw_ym )
      {
        cerr << "Error: two or more <year_month> tags found in <book_entry>."
             << endl ;
        exit ( 1 ) ;
      }
      sw_ym = true ;
      ym = new YearMonth ( n_ch ) ;
    }
    else
    {
      cerr << "Error: tag <" << n_ch -> Value ()
           << "> tag not allowedfound in <book_entry>." << endl ;
      exit ( 1 ) ;
    }
  }
  if ( ! ( sw_title && sw_ym && sw_pb && sw_pa )
       || authors . size () == 0 )
  {
    cerr << "Error: incomplete information found in tag <book_entry>."
         << endl ;
    exit ( 1 ) ;
  }
}



// ----
// Writing out authors.
// 著者一覧を書き出す。
// ----
void
BookEntry :: printAuthors
( ostream & ostr )
  const
{
  if ( authors . size () == 1 )
  {
    ostr << * ( authors [ 0 ] ) ;
  }
  else
  {
    vector < Author * > :: const_iterator ia = authors . end () ;
    -- ia ;
    vector < Author * > :: const_iterator ib = authors . begin () ;
    for ( vector < Author * > :: const_iterator ii
            = authors . begin () ;
          ii != authors . end () ; ++ ii )
    {
      if ( ii != ib )
      {
        if ( ii != ia )
        {
          ostr << ", " ;
        }
        else
        {
          ostr << " and " ;
        }
      }
      ostr << ( * ( * ii ) ) ;
    }
  }
}



// ----
// Writing out title.
// タイトルを書き出す。
// ----
void
BookEntry :: printTitle
( ostream & ostr )
  const
{
  ostr << title_str ;
}



// ----
// Writing out the date (month / year) of publication.
// 出版年月を出力する。
// ----
void
BookEntry :: printPublishedYearMonth
( ostream & ostr )
  const
{
  ostr << ( * ym ) ;
}



// ----
// Writing other bibliography information.
// 他の書誌情報を出力する。
// ----
void
BookEntry :: printOtherBibInfo
( ostream & ostr )
  const
{
  ostr << publisher_str << ", " << paddr_str ;
}



// ----
// Writing out contents of class BookEntry.
// BookEntry クラスの内容の書き出し。
// ----
ostream &
operator<<
( ostream & ostr ,
  const BookEntry & be_out )
{
  be_out . printAuthors ( ostr ) ;
  ostr << ": " ;
  be_out . printTitle ( ostr ) ;
  ostr << ", " ;
  be_out . printOtherBibInfo ( ostr ) ;
  ostr << ", " ;
  be_out . printPublishedYearMonth ( ostr ) ;
  ostr << endl ;
  return ostr ;
}

ostream &
BookEntry :: outputToStream
( ostream & ostr )
  const
{
  ostr << ( * this ) ;
  return ostr ;
}



// ----
// Destructor of class BookEntry.
// BookEntry クラスのデストラクタ.
// ----
BookEntry :: ~BookEntry
()
{
  for ( vector < Author * > :: iterator ii = authors . begin () ;
        ii != authors . end () ; ++ ii )
  {
    delete ( * ii ) ;
  }
  if ( ym )
    delete ym ;
}
