// -*- C++ -*-

#include <iostream>
#include <cstdlib>
#include <vector>
#include <string>
#include <sstream>
#include "reflist2-JournalPaperEntry.h"

using std :: ostream ;
using std :: vector ;
using std :: cerr ;
using std :: endl ;
using std :: string ;
using std :: istringstream ;


// ----
// Constructor of class JournalPaperEntry.
// JournalPaperEntry クラスのコンストラクタ.
// ----
JournalPaperEntry :: JournalPaperEntry
( const XMLNodePointer & n_in )
  : authors ()
{
  bool sw_title = false ;
  bool sw_ym = false ;
  bool sw_jn = false ;
  bool sw_vn = false ;
  bool sw_pg = false ;
  for ( const TiXmlNode * n_ch = n_in -> FirstChild () ;
        n_ch ; n_ch = n_ch -> NextSibling () )
  {
    if ( n_ch -> Type () != TiXmlNode :: ELEMENT )
    {
      continue ;
    }
    if ( string ( n_ch -> Value () ) == "title" )
    {
      if ( sw_title )
      {
        cerr << "Error: two or more <title> tags found in <journal_paper_entry>."
             << endl ;
        exit ( 1 ) ;
      }
      sw_title = true ;
      bool sw_tf = false ;
      for ( const TiXmlAttribute * t_a
              = n_ch -> ToElement () -> FirstAttribute () ;
            t_a ; t_a = t_a -> Next () )
      {
        if ( string ( t_a -> Name () ) == "name" )
        {
          if ( sw_tf )
          {
            cerr << "Error: duplicate \"name\" attribute to tag <title>"
                 << "... internal error?" << endl ;
            exit ( 1 ) ;
          }
          sw_tf = true ;
          title_str = t_a -> ValueStr () ;
        }
        else
        {
          cerr << "Error: attribute \"" << t_a -> Name ()
               << "\" not allowed in tag <title>" << endl ;
          exit ( 1 ) ;
        }
      }
    }
    else if ( string ( n_ch -> Value () ) == "author" )
    {
      Author * a_x = new Author ( n_ch ) ;
      authors . push_back ( a_x ) ;
    }
    else if ( string ( n_ch -> Value () ) == "journal" )
    {
      if ( sw_jn )
      {
        cerr << "Error: two or more <journal> tags found in <journal_paper_entry>."
             << endl ;
        exit ( 1 ) ;
      }
      sw_jn = true ;
      bool sw_bf = false ;
      for ( const TiXmlAttribute * t_a
              = n_ch -> ToElement () -> FirstAttribute () ;
            t_a ; t_a = t_a -> Next () )
      {
        if ( string ( t_a -> Name () ) == "name" )
        {
          if ( sw_bf )
          {
            cerr << "Error: duplicate \"name\" attribute to tag <journal>"
                 << "... internal error?" << endl ;
            exit ( 1 ) ;
          }
          sw_bf = true ;
          journal_str = t_a -> ValueStr () ;
        }
        else
        {
          cerr << "Error: attribute \"" << t_a -> Name ()
               << "\" not allowed in tag <journal>" << endl ;
          exit ( 1 ) ;
        }
      }
    }
    else if ( string ( n_ch -> Value () ) == "volume_number" )
    {
      if ( sw_vn )
      {
        cerr << "Error: two or more <volume_number> tags found in <journal_paper_entry>."
             << endl ;
        exit ( 1 ) ;
      }
      sw_vn = true ;
      bool sw_vol = false ;
      bool sw_num = false ;
      for ( const TiXmlAttribute * t_a
              = n_ch -> ToElement () -> FirstAttribute () ;
            t_a ; t_a = t_a -> Next () )
      {
        if ( string ( t_a -> Name () ) == "volume" )
        {
          if ( sw_vol )
          {
            cerr << "Error: duplicate \"volume\" attribute to tag "
                 << "<volume_number>... internal error?" << endl ;
            exit ( 1 ) ;
          }
          sw_vol = true ;
          istringstream ix ( t_a -> ValueStr () ) ;
          ix >> j_vol ;
        }
        else if ( string ( t_a -> Name () ) == "number" )
        {
          if ( sw_num )
          {
            cerr << "Error: duplicate \"number\" attribute to tag "
                 << "<volume_number>... internal error?" << endl ;
            exit ( 1 ) ;
          }
          sw_num = true ;
          istringstream iy ( t_a -> ValueStr () ) ;
          iy >> j_num ;
        }
        else
        {
          cerr << "Error: attribute \"" << t_a -> Name ()
               << "\" not allowed in tag <volume_number>" << endl ;
          exit ( 1 ) ;
        }
      }
      if ( ! ( sw_vol && sw_num ) )
      {
        cerr << "Error: not all attributes are specified in tag "
             << "<volume_number>" << endl ;
        exit ( 1 ) ;
      }
    }
    else if ( string ( n_ch -> Value () ) == "pages" )
    {
      if ( sw_pg )
      {
        cerr << "Error: two or more <pages> tags found in <journal_paper_entry>."
             << endl ;
        exit ( 1 ) ;
      }
      sw_pg = true ;
      bool sw_ps = false ;
      bool sw_pe = false ;
      for ( const TiXmlAttribute * t_a
              = n_ch -> ToElement () -> FirstAttribute () ;
            t_a ; t_a = t_a -> Next () )
      {
        if ( string ( t_a -> Name () ) == "start" )
        {
          if ( sw_ps )
          {
            cerr << "Error: duplicate \"start\" attribute to tag "
                 << "<pages>... internal error?" << endl ;
            exit ( 1 ) ;
          }
          sw_ps = true ;
          istringstream is ( t_a -> ValueStr () ) ;
          is >> page_start ;
        }
        else if ( string ( t_a -> Name () ) == "end" )
        {
          if ( sw_pe )
          {
            cerr << "Error: duplicate \"end\" attribute to tag "
                 << "<pages>... internal error?" << endl ;
            exit ( 1 ) ;
          }
          sw_pe = true ;
          istringstream ie ( t_a -> ValueStr () ) ;
          ie >> page_end ;
        }
        else
        {
          cerr << "Error: attribute \"" << t_a -> Name ()
               << "\" not allowed in tag <pages>" << endl ;
          exit ( 1 ) ;
        }
      }
      if ( ! ( sw_ps && sw_pe ) )
      {
        cerr << "Error: not all attributes are specified in tag "
             << "<pages>" << endl ;
        exit ( 1 ) ;
      }
    }
    else if ( string ( n_ch -> Value () ) == "year_month" )
    {
      if ( sw_ym )
      {
        cerr << "Error: two or more <year_month> tags found in <journal_paper_entry>."
             << endl ;
        exit ( 1 ) ;
      }
      sw_ym = true ;
      ym = new YearMonth ( n_ch ) ;
    }
    else
    {
      cerr << "Error: tag <" << n_ch -> Value ()
           << "> tag not allowedfound in <journal_paper_entry>." << endl ;
      exit ( 1 ) ;
    }
  }
  if ( ! ( sw_title && sw_ym && sw_jn && sw_vn && sw_pg )
       || authors . size () == 0 )
  {
    cerr << "Error: incomplete information found in tag <journal_paper_entry>."
         << endl ;
    exit ( 1 ) ;
  }
}



// ----
// Writing out authors.
// 著者一覧を書き出す。
// ----
void
JournalPaperEntry :: printAuthors
( ostream & ostr )
  const
{
  if ( authors . size () == 1 )
  {
    ostr << * ( authors [ 0 ] ) ;
  }
  else
  {
    vector < Author * > :: const_iterator ia = authors . end () ;
    -- ia ;
    vector < Author * > :: const_iterator ib = authors . begin () ;
    for ( vector < Author * > :: const_iterator ii
            = authors . begin () ;
          ii != authors . end () ; ++ ii )
    {
      if ( ii != ib )
      {
        if ( ii != ia )
        {
          ostr << ", " ;
        }
        else
        {
          ostr << " and " ;
        }
      }
      ostr << ( * ( * ii ) ) ;
    }
  }
}



// ----
// Writing out title.
// タイトルを書き出す。
// ----
void
JournalPaperEntry :: printTitle
( ostream & ostr )
  const
{
  ostr << title_str ;
}



// ----
// Writing out the date (month / year) of publication.
// 出版年月を出力する。
// ----
void
JournalPaperEntry :: printPublishedYearMonth
( ostream & ostr )
  const
{
  ostr << ( * ym ) ;
}



// ----
// Writing other bibliography information.
// 他の書誌情報を出力する。
// ----
void
JournalPaperEntry :: printOtherBibInfo
( ostream & ostr )
  const
{
  ostr << journal_str << ", Vol. " << j_vol << ", No. " << j_num << ", " ;
  if ( page_start == page_end )
  {
    ostr << "p. " << page_start ;
  }
  else
  {
    ostr << "pp. " << page_start << " - " << page_end ;
  }
}



// ----
// Writing out contents of class JournalPaperEntry.
// JournalPaperEntry クラスの内容の書き出し。
// ----
ostream &
operator<<
( ostream & ostr ,
  const JournalPaperEntry & be_out )
{
  be_out . printAuthors ( ostr ) ;
  ostr << ": " ;
  be_out . printTitle ( ostr ) ;
  ostr << ", " ;
  be_out . printOtherBibInfo ( ostr ) ;
  ostr << ", " ;
  be_out . printPublishedYearMonth ( ostr ) ;
  ostr << endl ;
  return ostr ;
}


ostream &
JournalPaperEntry :: outputToStream
( ostream & ostr )
  const
{
  ostr << ( * this ) ;
  return ostr ;
}




// ----
// Destructor of class JournalPaperEntry.
// JournalPaperEntry クラスのデストラクタ.
// ----
JournalPaperEntry :: ~JournalPaperEntry
()
{
  for ( vector < Author * > :: iterator ii = authors . begin () ;
        ii != authors . end () ; ++ ii )
  {
    delete ( * ii ) ;
  }
  if ( ym )
    delete ym ;
}
