// This may look like C code, but it is really -*- C++ -*-
// RTSS --- Railway Total System Simulator
// (c) TAKAGI Ryo (at Kogakuin University)
// Expandable.hh --- template classes SimpleExpandable & Expandable
// -----
// ChangeLog:
// 2009. 7. 10
//  Using <map> instead of <vector> in ObjectCreatorsList subclasses.
// 2007. 11. 19
//  Merged into the source of RTSS.
// 2004. 6. 5
//  Original date on the file --- OOMTS of Birmingham University
// -----

// -----
// Remark:
//  This file makes it easy to expand the program at a later date.
// Classes, structs and typedefs:
//  template <PP> class SimpleExpandable
//  template <PP, CA> class Expandable
// -----

#ifndef Expandable_H
#define Expandable_H

#include <cstdlib>
#include <string>
#include <map>
#include <iostream>

using std :: map ;
using std :: string ;
using std :: cerr ;
using std :: endl ;

/******************************************************************
 * class SimpleExpandable
 ******************************************************************/

template < typename PP >
class SimpleExpandable
{

public :

  // Various classes needed
  class ObjectCreator
  {
  public :
    virtual PP * create () const = 0 ;
    void add_to_creators_list ( const string & ) ;
  } ;

  friend class ObjectCreator ;


//   struct ObjectCreatorEntry
//   {
//     ObjectCreatorEntry
//     ( const ObjectCreator * , const char * ) ;
//     const ObjectCreator * creator ;
//     std :: string object_type ;
//   } ;

  class ObjectCreatorsList
    : public map < string , ObjectCreator * >
  {
  public:
    ObjectCreatorsList ()
      : map < string , ObjectCreator * > () {}
  } ;

private :

  static ObjectCreatorsList * & object_creators () ;

  // Add another object creator
  static void add_object_creator ( const string & , ObjectCreator * ) ;

public :

  // Constructor
  SimpleExpandable () {} ;

  // Create object of type PP, return pointer
  static PP * create ( const char * ) ;
  static PP * create ( const string & ) ;
} ;


/******************************************************************
 * class Expandable
 ******************************************************************/

template < typename PP , typename CA >
class Expandable
{

public :

  // Various classes needed
  class ObjectCreator
  {
  public :
    virtual PP * create () const = 0 ;
    virtual PP * create ( CA const & ) const = 0 ;
    void add_to_creators_list ( const string & ) ;
  } ;

  friend class ObjectCreator ;


//   struct ObjectCreatorEntry
//   {
//     ObjectCreatorEntry
//     ( const ObjectCreator * , const char * ) ;
//     const ObjectCreator * creator ;
//     std :: string object_type ;
//   } ;

  class ObjectCreatorsList
    : public map < string , ObjectCreator * >
  {
  public :
    ObjectCreatorsList ()
      : map < string , ObjectCreator * > () {}
  } ;

private :

  static ObjectCreatorsList * & object_creators () ;

  // Add another object creator
  static void add_object_creator ( const string & , ObjectCreator * ) ;

public :

  // Constructor
  Expandable () {} ;

  // Create object of type PP, return pointer
  static PP * create ( const char * ) ;
  static PP * create ( const string & ) ;
  static PP * create ( const char * , CA const & ) ;
  static PP * create ( const string & , CA const & ) ;
} ;


/******************************************************************
 *  Function definitions
 ******************************************************************/

template < typename PP >
typename SimpleExpandable < PP > :: ObjectCreatorsList * &
SimpleExpandable < PP > :: object_creators ()
{
  static typename SimpleExpandable < PP > :: ObjectCreatorsList *
    x = new typename SimpleExpandable < PP > :: ObjectCreatorsList ;
  return x ;
}



template < typename PP , typename CA >
typename Expandable < PP , CA > :: ObjectCreatorsList * &
Expandable < PP , CA > :: object_creators ()
{
  static typename Expandable < PP , CA > :: ObjectCreatorsList *
    x = new typename Expandable < PP , CA > :: ObjectCreatorsList ;
  return x ;
}



// template < typename PP >
// SimpleExpandable < PP > :: ObjectCreatorEntry :: ObjectCreatorEntry
// ( const typename SimpleExpandable < PP > :: ObjectCreator * creator_in ,
//   const char * object_type_in )
// {
//   creator = creator_in ;
//   object_type = object_type_in ;
// }



// template < typename PP , typename CA >
// Expandable < PP , CA > :: ObjectCreatorEntry :: ObjectCreatorEntry
// ( const typename Expandable < PP , CA > :: ObjectCreator * creator_in ,
//   const char * object_type_in )
// {
//   creator = creator_in ;
//   object_type = object_type_in ;
// }



template < typename PP >
PP *
SimpleExpandable < PP > :: create
( const string & _oc_in )
{
  ObjectCreatorsList * creators
    = SimpleExpandable < PP > :: object_creators () ;
  typename map < string , ObjectCreator * > :: iterator
    ii = creators -> find ( _oc_in ) ;
  if ( ii == creators -> end () )
  {
    // ### We should consider throwing an error.
    cerr << "Error: Creator for \"" << _oc_in << "\" not found."
         << endl ;
    exit ( 1 ) ;
  }
  else
  {
    return ii -> second -> create () ;
  }
}



template < typename PP , typename CA >
PP *
Expandable < PP , CA > :: create
( const string & _oc_in ,
  CA const & _ca_in )
{
  ObjectCreatorsList * creators
    = Expandable < PP , CA > :: object_creators () ;
  typename map < string , ObjectCreator * > :: iterator 
    ii = creators -> find ( _oc_in ) ;
  if ( ii == creators -> end () )
  {
    // ### We should consider throwing an error.
    cerr << "Error: Creator for \"" << _oc_in << "\" not found."
         << endl ;
    exit ( 2 ) ;
  }
  else
  {
    return ii -> second -> create ( _ca_in ) ;
  }
}



template < typename PP , typename CA >
PP *
Expandable < PP , CA > :: create
( const string & _oc_in )
{
  ObjectCreatorsList * creators
    = Expandable < PP , CA > :: object_creators () ;
  typename map < string , ObjectCreator * > :: iterator 
    ii = creators -> find ( _oc_in ) ;
  if ( ii == creators -> end () )
  {
    // ### We should consider throwing an error.
    cerr << "Error: Creator for \"" << _oc_in << "\" not found."
         << endl ;
    exit ( 2 ) ;
  }
  else
  {
    return ii -> second -> create () ;
  }
}



template < typename PP >
PP *
SimpleExpandable < PP > :: create
( const char * object_type_in )
{
  return create ( string ( object_type_in ) ) ;
}



template < typename PP , typename CA >
PP *
Expandable < PP , CA > :: create
( const char * object_type_in ,
  CA const & ca_in )
{
  return create ( string ( object_type_in ) , ca_in ) ;
}



template < typename PP , typename CA >
PP *
Expandable < PP , CA > :: create
( const char * object_type_in )
{
  return create ( string ( object_type_in ) ) ;
}



template < typename PP >
void
SimpleExpandable < PP > :: add_object_creator
( const string & _n_in ,
  typename SimpleExpandable < PP > :: ObjectCreator * _oc_in )
{
  ObjectCreatorsList * creators
    = SimpleExpandable < PP > :: object_creators () ;
  if ( creators -> find ( _n_in ) == creators -> end () )
  {
    ( * creators ) [ _n_in ] = _oc_in ;
  }
  else
  {
    cerr << "Error: there already exists a creator with key \""
         << _n_in << "\"" << endl ;
    exit ( 1 ) ;
  }
}



template < typename PP , typename CA >
void
Expandable < PP , CA > :: add_object_creator
( const string & _n_in ,
  typename Expandable < PP , CA > :: ObjectCreator * _oc_in )
{
  ObjectCreatorsList * creators
    = Expandable < PP , CA > :: object_creators () ;
  if ( creators -> find ( _n_in ) == creators -> end () )
  {
    ( * creators ) [ _n_in ] = _oc_in ;
  }
  else
  {
    cerr << "Error: there already exists a creator with key \""
         << _n_in << "\"" << endl ;
    exit ( 1 ) ;
  }
}



template < typename PP >
void
SimpleExpandable < PP > :: ObjectCreator :: add_to_creators_list
( const string & _oc_in )
{
  SimpleExpandable < PP > :: add_object_creator ( _oc_in , this ) ;
}


template < typename PP , typename CA >
void
Expandable < PP , CA > :: ObjectCreator :: add_to_creators_list
( const string & _oc_in )
{
  Expandable < PP , CA > :: add_object_creator ( _oc_in , this ) ;
}

#endif	// Expandable_H
