// -*- C++ -*-

// ダイクストラ法のデモ

#include "RS_FHeap.hh"

#define RS_DIJKSTRA_PRINT

#include "RS_Dijkstra.hh"
#include "Sujic_Weight.hh"
#include "RS_Object.hh"
#include "X_SpaceTimeNode.hh"
#include "X_Station.hh"
#include "X_Train.hh"
#include "X_SpaceTimeBranch.hh"
#include "X_CycleTime_Initialiser.hh"
#include "rtss_path.hh"

#include <cstdlib>
#include <iostream>
#include <sstream>
#include <vector>
#include <string>
#include <map>
#include <unistd.h>

using std :: cerr ;
using std :: endl ;
using std :: ostream ;
using std :: ostringstream ;
using std :: string ;
using std :: sort ;
using std :: map ;




class X_Passengers_F1
{
  X_SpaceTimeBranch const * _stb_st ;
  X_Station_Base const * _sta_dest ;
  int _t_ ;
  int _cd_ ;
  double _num_passen ;

public :
  X_Passengers_F1
  ( X_SpaceTimeBranch const * _stb_in ,
    X_Station_Base const * _sta_d_in , int _t_in , int _cd_in , double _np_in )
    : _stb_st ( _stb_in ) , _sta_dest ( _sta_d_in ) ,
      _t_ ( _t_in ) , _cd_ ( _cd_in ) , _num_passen ( _np_in ) {}

  X_SpaceTimeBranch const * getStartSTB () const { return _stb_st ; }
  double getPassengerNumber () const { return _num_passen ; }
  int getTime () const { return _t_ ; }
  int getCycleDelay () const { return _cd_ ; }
  X_Station_Base const * getDestination () const { return _sta_dest ; }
} ;



int X_SpaceTimeNode :: CYCLE_TIME = 0 ;
bool X_SpaceTimeNode :: DO_CYCLIC_SIMULATION = true ;




int
main
( int argc ,
  char * * argv )
{
  string current_directory = "" ;
  string current_drive = "" ;
  bool is_backslash = false ;

  // -----
  // Prepare object containers and an XML reader.
  // オブジェクトコンテナとXMLリーダオブジェクトを準備する。
  // -----
  RS_XmlReader xread ;
  X_CTI_Container xc_c ;
  X_Station_Container xs_c ;
  X_Train_Container xt_c ;
  xread [ "sujic_cycletime" ] = & xc_c ;
  xread [ "sujic_station" ] = & xs_c ;
  xread [ "sujic_train" ] = & xt_c ;

  // -----
  // Get the "current working directory" for this program.
  // このプログラムのカレントディレクトリを得る。
  // -----
  char * curpath = getcwd ( NULL , 0 ) ;
  if ( curpath == NULL )
  {
    cerr << "Error: Getting present working directory failed" << endl ;
    exit ( 1 ) ;
  }
  else
  {
    cerr << "CWD: " << curpath << endl ;
  }
  current_directory = curpath ;
  current_drive = "" ;

  // -----
  // Check if the directory is "Windows" style. If windows, then separate the
  // drive name and the file path.
  // -----
  RTSS_Path_String check_win_path ( curpath ) ;
  free ( curpath ) ;
  is_backslash = check_win_path . isWindowsStyleFullPath ( current_drive ) ;
  if ( is_backslash )
  {
    current_directory = check_win_path ;
    cerr << "Windows path, dir = " << current_directory << ", drive = "
          << current_drive << "!" << endl ;
  }
  else
  {
    cerr << "Unix path, dir = " << current_directory << ", drive = "
          << current_drive << "!" << endl ;
  }

  // ----
  // There must be at least one command-line argument.
  // ----
  if ( argc == 1 )
  {
    cerr << "Error: There must be at least one command-line argument."
         << endl ;
    exit ( 1 ) ;
  }

  // -----
  // Assume the command-line arguments are file names, and process them.
  // -----
  vector < string > _files ;
  for ( -- argc ; argc ; -- argc )
  {
    RTSS_Path_String control_file ( * ++ argv ) ;
    // -----
    // Filename check (2 and thereafter) ... all processed here.
    // -----
    string controlfiledrive = "" ;
    string controlfiledir = "" ;
    string controlfilename = "" ;
    if ( ! control_file . processPath
         ( is_backslash , current_drive , current_directory ,
           controlfiledrive , controlfiledir , controlfilename ) )
    {
      cerr << "ERROR: unknown error in processPath (control file)" << endl ;
      exit ( 1 ) ;
    }
    current_drive = controlfiledrive ;
    current_directory = controlfiledir ;
    string _fname_in = controlfiledrive ;
    _fname_in += controlfiledir ;
    if ( is_backslash )
    {
      _fname_in += "\\" ;
    }
    else
    {
      _fname_in += "/" ;
    }
    _fname_in += controlfilename ;
    _files . push_back ( _fname_in ) ;
    cerr << "Control file: " << _fname_in << endl << " dir = "
         << current_directory << ", drive = " << current_drive << endl ;
    xread . openXmlDocument ( _fname_in , current_drive , current_directory ) ;
    do
    {
      xread . processElement () ;
    } while ( xread . hasNextElement () ) ;
  }
  cerr << endl << endl << "XML file Read Finished." << endl << endl ;

  // Create stations --- done through XML reading
  // Set OD demand data --- done through XML reading
  // Create trains --- done through XML reading

  //////////////////////////////////////////////////////////////////////

  // Set cycletime.
  X_SpaceTimeNode :: CYCLE_TIME = xc_c . getCycleTime () ;

  // Set times
  xt_c . setTimes ( & xs_c ) ;

  // ----
  // Sorting Space Time Nodes in X_Station objects.
  // ----
  for ( X_Station_Container :: iterator ii = xs_c . begin () ;
        ii != xs_c . end () ; ++ ii )
  {
    dynamic_cast < X_Station_Base * > ( ii -> second ) -> sortTimes () ;
  }

  // ----
  // Create and populate a vector of SpaceTimeBranch for starting the path
  // search, and a vector of passenger tokens.
  // ----
  map < X_STB_Key , X_SpaceTimeBranch * > _stb_map ;
  vector < X_SpaceTimeBranch * > _stb_st ;
  vector < X_Passengers_F1 * > _pas_f1 ;
  for ( X_Station_Container :: iterator ii = xs_c . begin () ;
        ii != xs_c . end () ; ++ ii )
  {
    X_Station_Base *
      p_ii = dynamic_cast < X_Station_Base * > ( ii -> second ) ;
    for ( vector < X_SpaceTimeNode * > :: const_iterator
            ix = p_ii -> firstNode () ;
          ix != p_ii -> lastNode () - 1 ; ++ ix )
    {
      // ----
      // Creation of a SpaceTimeBranch_Station object. This will be the
      // starting point of the path search using the Dijkstra's algorithm.
      // ----
      X_SpaceTimeNode const * _stn_s = * ix ;
      X_SpaceTimeNode const * _stn_e = * ( ix + 1 ) ;
      int _tx = _stn_e -> getTime () - _stn_s -> getTime () ;
      if ( _tx < 0 )
      {
        _tx += X_SpaceTimeNode :: CYCLE_TIME ;
      }
      if ( _tx < 0 || _tx >= X_SpaceTimeNode :: CYCLE_TIME )
      {
        cerr << "Error: internal error 1." << endl ;
        exit ( 1 ) ;
      }
      cerr << "Branch: " << _tx << ", " << _stn_s -> getName () << " ["
           << _stn_s -> getTime () << "/" << _stn_s -> getCycleDelay ()
           << "] -> " << _stn_e -> getName () << " ["
           << _stn_e -> getTime () << "/" << _stn_e -> getCycleDelay ()
           << "]" << endl ;
      X_SpaceTimeBranch * _sx = new X_SpaceTimeBranch
        ( _stn_s , _stn_e , Sujic_Weight ( _tx , 0 , 0 , _tx ) ) ;
      _sx -> setPassengerArrivalNode () ;
      _stb_st . push_back ( _sx ) ;
      if ( _stb_map . find ( X_STB_Key ( _stn_s , _stn_e ) )
           != _stb_map . end () )
      {
        cerr << "Error: already an element in _stb_map." << endl ;
        exit ( 2 ) ;
      }
      _stb_map [ X_STB_Key ( _stn_s , _stn_e ) ] = _sx ;

      // ----
      // Creation of X_Passengers_F1 objects. If there is a demand data
      // from this particular station, then a X_Passengers_F1 object per
      // destination station will be created. Each object will represent
      // ( D / C * T ) passengers, where D is the demand (number of
      // passengers) per cycle, C is the cycle time in [sec] and T is the
      // length in [sec] of the SpaceTimeBranch _stb_x.
      // ----
      X_SpaceTimeNode const * _n_s = _sx -> getStartSTN () ;
      X_SpaceTimeNode const * _n_e = _sx -> getEndSTN () ;
      int _t_s = _n_s -> getTime () ;
      int _cd_s = _n_s -> getCycleDelay () ;
      int _t_e = _n_e -> getTime () ;
      int _t_x = _t_e - _t_s ;
      _t_s += _t_x / 2 ;
      while ( _t_s >= X_SpaceTimeNode :: CYCLE_TIME )
      {
        _t_s -= X_SpaceTimeNode :: CYCLE_TIME ;
        ++ _cd_s ;
      }
      for ( X_OD_Station_Container :: const_iterator iy = p_ii -> firstOD () ;
            iy != p_ii -> lastOD () ; ++ iy )
      {
        X_OD_Station_Base const *
          x_od = dynamic_cast < X_OD_Station_Base const * > ( iy -> second ) ;
        if ( ! x_od )
        {
          cerr << "Error: dynamic cast failed in call for X_OD_Station."
               << endl ;
          exit ( 1 ) ;
        }
        double _dem_x = x_od -> getDemand () ;
        X_Station_Base const * _sta_dest = x_od -> getDestination () ;
        _dem_x *= _t_x ;
        _dem_x /= X_SpaceTimeNode :: CYCLE_TIME ;
        cerr << "  Passenger: \"" << _n_s -> getName () << "\"-|-\""
             << _n_e -> getName () << "\" -> \"" << _sta_dest -> getName ()
             << "\"" << endl << "    [" << _t_s << "/" << _cd_s << "]: "
             << _dem_x << " = " << _t_x << " / "
             << X_SpaceTimeNode :: CYCLE_TIME << " * " << x_od -> getDemand ()
             << endl ;
        X_Passengers_F1 *
          _x_pfi = new X_Passengers_F1
          ( _sx , _sta_dest , _t_s , _cd_s , _dem_x ) ;
        _pas_f1 . push_back ( _x_pfi ) ;
        _sx -> addInitiatingPassenger ( _x_pfi ) ;
      }
    }
    // ----
    // DO_CYCLIC_SIMULATION ** MUST ** be true...
    // ----
    if ( X_SpaceTimeNode :: DO_CYCLIC_SIMULATION )
    {
      vector < X_SpaceTimeNode * > :: const_iterator
        i_last = p_ii -> lastNode () - 1 ;
      vector < X_SpaceTimeNode * > :: const_iterator
        i_first = p_ii -> firstNode () ;
      // ----
      // Creation of a cyclic SpaceTimeBranch object.
      // ----
      X_SpaceTimeNode const * _stn_s = * i_last ;
      X_SpaceTimeNode const * _stn_e = * i_first ;
      int _tx = _stn_e -> getTime () - _stn_s -> getTime () ;
      if ( _tx < 0 )
      {
        _tx += X_SpaceTimeNode :: CYCLE_TIME ;
      }
      if ( _tx < 0 || _tx >= X_SpaceTimeNode :: CYCLE_TIME )
      {
        cerr << "Error: internal error 1." << endl ;
        exit ( 1 ) ;
      }
      cerr << "Branch: " << _tx << ", " << _stn_s -> getName () << " ["
           << _stn_s -> getTime () << "/" << _stn_s -> getCycleDelay ()
           << "] -> " << _stn_e -> getName () << " ["
           << _stn_e -> getTime () << "/" << _stn_e -> getCycleDelay ()
           << "]" << endl ;
      X_SpaceTimeBranch * _sx = new X_SpaceTimeBranch
        ( _stn_s , _stn_e , Sujic_Weight ( _tx , 0 , 0 , _tx ) ) ;
      _sx -> setPassengerArrivalNode () ;
      _stb_st . push_back ( _sx ) ;
      if ( _stb_map . find ( X_STB_Key ( _stn_s , _stn_e ) )
           != _stb_map . end () )
      {
        cerr << "Error: already an element in _stb_map." << endl ;
        exit ( 3 ) ;
      }
      _stb_map [ X_STB_Key ( _stn_s , _stn_e ) ] = _sx ;

      // ----
      // Creation of a "cyclic" X_Passengers_F1 object.
      // ----
      X_SpaceTimeNode const * _n_s = _sx -> getStartSTN () ;
      X_SpaceTimeNode const * _n_e = _sx -> getEndSTN () ;
      int _t_s = _n_s -> getTime () ;
      int _cd_s = _n_s -> getCycleDelay () ;
      int _t_e = _n_e -> getTime () ;
      int _t_x = _t_e - _t_s + X_SpaceTimeNode :: CYCLE_TIME ;
      _t_s += _t_x / 2 ;
      while ( _t_s >= X_SpaceTimeNode :: CYCLE_TIME )
      {
        _t_s -= X_SpaceTimeNode :: CYCLE_TIME ;
        ++ _cd_s ;
      }
      for ( X_OD_Station_Container :: const_iterator iy = p_ii -> firstOD () ;
            iy != p_ii -> lastOD () ; ++ iy )
      {
        X_OD_Station_Base const *
          x_od = dynamic_cast < X_OD_Station_Base const * > ( iy -> second ) ;
        if ( ! x_od )
        {
          cerr << "Error: dynamic cast failed in call for X_OD_Station."
               << endl ;
          exit ( 1 ) ;
        }
        double _dem_x = x_od -> getDemand () ;
        X_Station_Base const * _sta_dest = x_od -> getDestination () ;
        _dem_x *= _t_x ;
        _dem_x /= X_SpaceTimeNode :: CYCLE_TIME ;
        cerr << "  Passenger: \"" << _n_s -> getName () << "\"-|-\""
             << _n_e -> getName () << "\" -> \"" << _sta_dest -> getName ()
             << "\"" << endl << "    [" << _t_s << "/" << _cd_s << "]: "
             << _dem_x << " = " << _t_x << " / "
             << X_SpaceTimeNode :: CYCLE_TIME << " * " << x_od -> getDemand ()
             << endl ;
        X_Passengers_F1 *
          _x_pfi = new X_Passengers_F1
          ( _sx , _sta_dest , _t_s , _cd_s , _dem_x ) ;
        _pas_f1 . push_back ( _x_pfi ) ;
        _sx -> addInitiatingPassenger ( _x_pfi ) ;
      }
    }
  }

  // ----
  // Create and populate the X_SpaceTimeBranch vector. This will contain
  // branches other than those linking X_SpaceTimeNodes at the same station.
  // ----
  vector < X_SpaceTimeBranch * > _stb_n ;

  // ----
  // Between every SpaceTimeNode at station and station exit.
  // ----
  for ( X_Station_Container :: iterator ii = xs_c . begin () ;
        ii != xs_c . end () ; ++ ii )
  {
    X_Station_Base *
      p_ii = dynamic_cast < X_Station_Base * > ( ii -> second ) ;
    for ( vector < X_SpaceTimeNode * > :: const_iterator
            ix = p_ii -> firstNode () ;
          ix != p_ii -> lastNode () ; ++ ix )
    {
      cerr << "Branch (station-exit): " << ( * ix ) -> getName () << " -> "
           << p_ii -> getExit () -> getName () << endl ;
      X_SpaceTimeBranch * _sx = new X_SpaceTimeBranch
        ( * ix , p_ii -> getExit () ,
          Sujic_Weight ( 0 , 0 , 0 , 0 ) ) ;
      _stb_n . push_back ( _sx ) ;
      if ( _stb_map . find ( X_STB_Key ( * ix , p_ii -> getExit () ) )
           != _stb_map . end () )
      {
        cerr << "Error: already an element in _stb_map." << endl ;
        exit ( 1 ) ;
      }
      _stb_map [ X_STB_Key ( * ix , p_ii -> getExit () ) ] = _sx ;
    }
  }

  // ----
  // Between passenger arrival and exit of the same station (meaningless, but
  // necessary (?) in calculation).
  // ----
  for ( vector < X_SpaceTimeBranch * > :: iterator
          istb = _stb_st . begin () ; istb != _stb_st . end () ; ++ istb )
  {
    X_SpaceTimeNode const *
      _stb_ptr = ( * istb ) -> getPassengerArrivalNode () ;
    X_SpaceTimeNode const *
      _own_exit
      = ( * istb ) -> getStartSTN () -> getStation () -> getExit () ;
    cerr << "Branch (dummy): " << _stb_ptr -> getName ()
         << " -> " << _own_exit -> getName () << endl ;
    X_SpaceTimeBranch * _sx = new X_SpaceTimeBranch
      ( _stb_ptr , _own_exit , Sujic_Weight ( 0 , 0 , 0 , 0 ) ) ;
    if ( _stb_map . find ( X_STB_Key ( _stb_ptr , _own_exit ) )
         != _stb_map . end () )
    {
      cerr << "Error: already an element in _stb_map." << endl ;
      exit ( 4 ) ;
    }
    _stb_map [ X_STB_Key ( _stb_ptr , _own_exit ) ] = _sx ;
  }

  // ----
  // Creating train branches.
  // ----
  for ( X_Train_Container :: iterator ii = xt_c . begin () ;
        ii != xt_c . end () ; ++ ii )
  {
    X_Train_Base const *
      _trn = dynamic_cast < X_Train_Base const * > ( ii -> second ) ;
    if ( ! _trn )
    {
      cerr << "Error: dynamic cast to X_Train failed" << endl ;
      exit ( 101 ) ;
    }
    for ( vector < X_SpaceTimeNode * > :: const_iterator ix
            = _trn -> firstNode () ;
          ix != _trn -> lastNode () - 1 ; ++ ix )
    {
      X_SpaceTimeNode const * _nns = * ix ;
      X_SpaceTimeNode const * _nne = * ( ix + 1 ) ;
      int _ts = ( * ix ) -> getTime () ;
      int _cds = ( * ix ) -> getCycleDelay () ;
      int _te = ( * ( ix + 1 ) ) -> getTime () ;
      int _cde = ( * ( ix + 1 ) ) -> getCycleDelay () ;
      int _tb = ( _te + _cde * X_SpaceTimeNode :: CYCLE_TIME )
        - ( _ts + _cds * X_SpaceTimeNode :: CYCLE_TIME ) ;
      cerr << "Branch (train): " << _nns -> getName ()
           << " -> " << _nne -> getName () << endl ;
      X_SpaceTimeBranch * _sx = new X_SpaceTimeBranch
        ( _nns , _nne , Sujic_Weight ( _tb , 0 , 0 , 0 ) ) ;
      _stb_n . push_back ( _sx ) ;
      if ( _stb_map . find ( X_STB_Key ( _nns , _nne ) )
           != _stb_map . end () )
      {
        cerr << "Error: already an element in _stb_map." << endl ;
        exit ( 5 ) ;
      }
      _stb_map [ X_STB_Key ( _nns , _nne ) ] = _sx ;
    }

    for ( vector < X_SpaceTimeNode * > :: const_iterator
            ix = _trn -> firstNode () ; ix != _trn -> lastNode () ; ++ ix )
    {
      X_SpaceTimeNode const * _nns = * ix ;
      X_SpaceTimeNode const * _nne = _nns ;
      if ( _nns -> isDeparting () )
      {
        _nns = _nne -> getStation () -> getCorrespondingNode ( _nne ) ;
      }
      else
      {
        _nne = _nns -> getStation () -> getCorrespondingNode ( _nns ) ;
      }
      cerr << "Branch (transfer): " << _nns -> getName ()
           << " -> " << _nne -> getName () << endl ;
      X_SpaceTimeBranch * _sx = new X_SpaceTimeBranch
        ( _nns , _nne , Sujic_Weight ( 0 , 1 , 0 , 0 ) ) ;
      _stb_n . push_back ( _sx ) ;
      if ( _stb_map . find ( X_STB_Key ( _nns , _nne ) )
           != _stb_map . end () )
      {
        cerr << "Error: already an element in _stb_map." << endl ;
        exit ( 6 ) ;
      }
      _stb_map [ X_STB_Key ( _nns , _nne ) ] = _sx ;
    }
  }

  // ----
  // For each SpaceTimeBranch in vector _stb_st, perform path search
  // using a Dijkstra's algorithm solver. Solvers must be kept on the memory
  // after the algorithm has been run.
  // ----
  map < X_SpaceTimeBranch const * ,
    RS_Dijkstra < Sujic_Weight , X_SpaceTimeNode const * > * > _dk_solvers ;

  for ( vector < X_SpaceTimeBranch * > :: iterator
          istb = _stb_st . begin () ; istb != _stb_st . end () ; ++ istb )
  {
    // ----
    // Setting names.
    // ----
    string _stb_name = ( * istb ) -> getMiddleName () ;
    X_SpaceTimeNode const *
      _stb_ptr = ( * istb ) -> getPassengerArrivalNode () ;
    X_SpaceTimeNode const * _own_exit
      = ( * istb ) -> getStartSTN () -> getStation () -> getExit () ;

    // ----
    // Creating a Dijkstra's Algorithm solver.
    // ----
    RS_Dijkstra < Sujic_Weight , X_SpaceTimeNode const * > *
      _x_slv = new RS_Dijkstra < Sujic_Weight , X_SpaceTimeNode const * >
      ( Sujic_Weight ( LARGENUM , 0 , 0 , 0 ) ) ;
    _dk_solvers [ * istb ] = _x_slv ;

    // ----
    // Creating nodes in the Dijkstra's algorithm solver.
    // ----
    for ( X_Station_Container :: iterator ii = xs_c . begin () ;
          ii != xs_c . end () ; ++ ii )
    {
      X_Station_Base *
        p_ii = dynamic_cast < X_Station_Base * > ( ii -> second ) ;
      for ( vector < X_SpaceTimeNode * > :: const_iterator ix
              = p_ii -> firstNode () ;
            ix != p_ii -> lastNode () ; ++ ix )
      {
        _x_slv -> setNode ( ( * ix ) , ( * ix ) -> getName () ) ;
      }
      X_SpaceTimeNode const * _exit = p_ii -> getExit () ;
      _x_slv -> setNode ( _exit , _exit -> getName () ) ;
    }

    // ----
    // Creating the node representing passengers' arrival at the originating
    // station.
    // ----
    _x_slv -> setNode ( _stb_ptr , _stb_name ) ;

    // ----
    // Trains must have their own nodes at stations.
    // ----
    for ( X_Train_Container :: iterator ii = xt_c . begin () ;
          ii != xt_c . end () ; ++ ii )
    {
      X_Train_Base const *
        _trn = dynamic_cast < X_Train_Base const * > ( ii -> second ) ;
      if ( ! _trn )
      {
        cerr << "Error: dynamic cast to X_Train failed" << endl ;
        exit ( 102 ) ;
      }
      for ( vector < X_SpaceTimeNode * > :: const_iterator
              ix = _trn -> firstNode () ; ix != _trn -> lastNode () ; ++ ix )
      {
        _x_slv -> setNode ( ( * ix ) , ( * ix ) -> getName () ) ;
      }
    }

    // ----
    // Creating branches at stations.
    // Note::: at stations, branch time length does not need consideration
    // of cycle delays.
    // ----
    for ( vector < X_SpaceTimeBranch * > :: iterator ii = _stb_st . begin () ;
          ii != _stb_st . end () ; ++ ii )
    {
      X_SpaceTimeNode const * _nns = ( * ii ) -> getStartSTN () ;
      X_SpaceTimeNode const * _nne = ( * ii ) -> getEndSTN () ;
      if ( ( * ii ) == ( * istb ) )
      {
        Sujic_Weight _wt_half = ( * ii ) -> getWeight () . getHalf () ;
        X_SpaceTimeNode const * _nnm = _stb_ptr ;
        _x_slv -> setBranch ( _nns , _nnm , _wt_half ) ;
        _x_slv -> setBranch
          ( _nnm , _nne , ( * ii ) -> getWeight () - _wt_half ) ;
        _x_slv -> setBranch
          ( _stb_ptr , _own_exit , Sujic_Weight ( 0 , 0 , 0 , 0 ) ) ;
      }
      else
      {
        _x_slv -> setBranch ( _nns , _nne , ( * ii ) -> getWeight () ) ;
      }
    }

    // ----
    // Creating other branches.
    // ----
    for ( vector < X_SpaceTimeBranch * > :: iterator ii = _stb_n . begin () ;
          ii != _stb_n . end () ; ++ ii )
    {
      X_SpaceTimeNode const * _nns = ( * ii ) -> getStartSTN () ;
      X_SpaceTimeNode const * _nne = ( * ii ) -> getEndSTN () ;
      _x_slv -> setBranch ( _nns , _nne , ( * ii ) -> getWeight () ) ;
    }

    // ----
    // Solver complete. Run the algorithm.
    // ----
    cerr << endl << "Solver complete: " << _stb_name << endl ;
    _x_slv -> run ( _stb_ptr , Sujic_Weight ( 0 , 0 , 0 , 0 ) ) ;

    // ----
    // Printing paths.
    // ----
    for ( X_Station_Container :: iterator ii = xs_c . begin () ;
          ii != xs_c . end () ; ++ ii )
    {
      X_Station_Base *
        p_ii = dynamic_cast < X_Station_Base * > ( ii -> second ) ;
      _x_slv -> printRoute ( p_ii -> getExit () ) ;
    }

    // ----
    // Putting passengers in containers.
    // ----
    for ( vector < X_Passengers_F1 const * > :: const_iterator
            ix = ( * istb ) -> firstInitialPassenger () ;
          ix != ( * istb ) -> lastInitialPassenger () ; ++ ix )
    {
      deque < X_SpaceTimeNode const * > _pth = _x_slv -> getRoute
        ( ( * ix ) -> getDestination () -> getExit () ) ;
      for ( deque < X_SpaceTimeNode const * > :: const_iterator
              ipth = _pth . begin () ; ipth != _pth . end () - 1 ; ++ ipth )
      {
        map < X_STB_Key , X_SpaceTimeBranch * > :: iterator
          ibr = _stb_map . find ( X_STB_Key ( * ipth , * ( ipth + 1 ) ) ) ;
        if ( ibr != _stb_map . end () )
        {
          ibr -> second -> addPassenger ( * ix ) ;
        }
      }
    }
  }

  // ----
  // Checking passengers in the containers.
  // ----
  cerr << endl ;
  for ( vector < X_SpaceTimeBranch * > :: iterator
          ii = _stb_n . begin () ; ii != _stb_n . end () ; ++ ii )
  {
    double _pas_s = 0 ;
    double _pas_e = 0 ;
    for ( vector < X_Passengers_F1 const * > :: const_iterator
            ix = ( * ii ) -> firstInitialPassenger () ;
          ix != ( * ii ) -> lastInitialPassenger () ; ++ ix )
    {
      _pas_e += ( * ix ) -> getPassengerNumber () ;
    }
    for ( vector < X_Passengers_F1 const * > :: const_iterator
            ix = ( * ii ) -> firstPassenger () ;
          ix != ( * ii ) -> lastPassenger () ; ++ ix )
    {
      _pas_s += ( * ix ) -> getPassengerNumber () ;
      _pas_e += ( * ix ) -> getPassengerNumber () ;
    }
    cerr << "Branch \"" << ( * ii ) -> getStartSTN () -> getName ()
         << "\" -> \"" << ( * ii ) -> getEndSTN () -> getName ()
         << "\": " << _pas_s << " -|- " << _pas_e << endl ;
    for ( vector < X_Passengers_F1 const * > :: const_iterator
            ix = ( * ii ) -> firstInitialPassenger () ;
          ix != ( * ii ) -> lastInitialPassenger () ; ++ ix )
    {
      cerr << "  [I] "
           << ( * ix ) -> getStartSTB () -> getStartStationName ()
           << ": [" << ( * ix ) -> getTime () << "/"
           << ( * ix ) -> getCycleDelay () << "] -> "
           << ( * ix ) -> getDestination () -> getName ()
           << ": " << ( * ix ) -> getPassengerNumber () << endl ;
    }
    for ( vector < X_Passengers_F1 const * > :: const_iterator
            ix = ( * ii ) -> firstPassenger () ;
          ix != ( * ii ) -> lastPassenger () ; ++ ix )
    {
      cerr << "  [P] "
           << ( * ix ) -> getStartSTB () -> getStartStationName ()
           << ": [" << ( * ix ) -> getTime () << "/"
           << ( * ix ) -> getCycleDelay () << "] -> "
           << ( * ix ) -> getDestination () -> getName ()
           << ": " << ( * ix ) -> getPassengerNumber () << endl ;
    }
  }
  for ( vector < X_SpaceTimeBranch * > :: iterator
          ii = _stb_st . begin () ; ii != _stb_st . end () ; ++ ii )
  {
    double _pas_s = 0 ;
    double _pas_e = 0 ;
    for ( vector < X_Passengers_F1 const * > :: const_iterator
            ix = ( * ii ) -> firstInitialPassenger () ;
          ix != ( * ii ) -> lastInitialPassenger () ; ++ ix )
    {
      _pas_e += ( * ix ) -> getPassengerNumber () ;
    }
    for ( vector < X_Passengers_F1 const * > :: const_iterator
            ix = ( * ii ) -> firstPassenger () ;
          ix != ( * ii ) -> lastPassenger () ; ++ ix )
    {
      _pas_s += ( * ix ) -> getPassengerNumber () ;
      _pas_e += ( * ix ) -> getPassengerNumber () ;
    }
    cerr << "Branch \"" << ( * ii ) -> getStartSTN () -> getName ()
         << "\" -> \"" << ( * ii ) -> getEndSTN () -> getName ()
         << "\": " << _pas_s << " -|- " << _pas_e << endl ;
    for ( vector < X_Passengers_F1 const * > :: const_iterator
            ix = ( * ii ) -> firstInitialPassenger () ;
          ix != ( * ii ) -> lastInitialPassenger () ; ++ ix )
    {
      cerr << "  [I] "
           << ( * ix ) -> getStartSTB () -> getStartStationName ()
           << ": [" << ( * ix ) -> getTime () << "/"
           << ( * ix ) -> getCycleDelay () << "] -> "
           << ( * ix ) -> getDestination () -> getName ()
           << ": " << ( * ix ) -> getPassengerNumber () << endl ;
    }
    for ( vector < X_Passengers_F1 const * > :: const_iterator
            ix = ( * ii ) -> firstPassenger () ;
          ix != ( * ii ) -> lastPassenger () ; ++ ix )
    {
      cerr << "  [P] "
           << ( * ix ) -> getStartSTB () -> getStartStationName ()
           << ": [" << ( * ix ) -> getTime () << "/"
           << ( * ix ) -> getCycleDelay () << "] -> "
           << ( * ix ) -> getDestination () -> getName ()
           << ": " << ( * ix ) -> getPassengerNumber () << endl ;
    }
  }
}
